/*-----------------------------------------------------------------------------------------
 *  This software is in the public domain, furnished "as is", without technical
 *  support, and with no warranty, express or implied, as to its usefulness for
 *  any purpose.
 *   
 *  MultiSwitch.ino 
 *  
 *  Multiswitch decoder for 4 switches. RC PWM signal length for each switch must be 
 *  half of the previous one.     
 *  
 *  This multiswitch based on the Yet another Multi Switch - YaMS implementation 
 *  of Dirk Wiedemann (www.rc-bastelbu.de), but is not using the servo lib for the 
 *  RC-Signal readings. The hardware information of the encoder should be taken from  
 *  the above referenzed web-location.
 *  For this 4 times MS i have used the following resistor values for the encoder:
 *  220Ohm, 470Ohm, 1000Ohm, 2200Ohm 
 *  The calibration steps of the RC input signals must be done as described 
 *  in PwmReceive.ino as well.
 *  
 *  Author: Volker Frauenstein 
 *  Date: 18/09/2025 Version: 1.5 minor bugfixing in debug data printing area
 *  
 *  Date: 12/05/2025 Version: 1.4 migration of PwmReceive (replace for RCPwmRead) including
 *                                support for new global RC_xxx and MS_xxx definitions from project header  
 *  Date: 28/12/2023 Version: 1.3 takeover for project soloe ruff 
 *                                MS_treshold values adapded to Skysport 4 RC system
 *  Date: 06/01/2021 Version: 1.2 functionality of hysteresis added for stability reasons 
 *                                of switch buttom recognition
 *  Date: 05/01/2021 Version: 1.1 pinout redefinition for migration to ESP32 architecture
 *  Date: 21/12/2020 Version: 1.0 implementation of multiswitch decoder for Arduino Nano 
 */
 
/* definition for pinout for the multiswitch  *****************************************************
const int msPin_a = xx;                                         // used SW internal (switchOn[MS_Func1]) 
const int msPin_b = xx;                                         // used SW internal (switchOn[MS_Func2]) 
const int msPin_c = xx;                                         // used SW internal (switchOn[MS_Func3])                           
const int msPin_d = xx;                                         // used SW internal (switchOn[MS_Func4])  */

// definitions for multiswitch functions **********************************************************
/* interface definitions to handle Multiswitch states -> transfered to project header to handle globaly
bool switchOn[4] = {false, false, false, false};                // one bit per switch: {MS_Func1, MS_Func2, MS_Func3, MS_Func4}
int last_state = 0;                                             // switch state of last cycle 
volatile float RC_MultiSwitch = 0;                              // RC input for Multiswitch normed from 1 to -1 */
int hysteresis = 0;                                             // counter for hysteresis 

// definitions for adjustable values **************************************************************
// Thresholds to separate every switch state from his neighbors, must be customized for real encoders
float MS_threshold [17] = { -1.0625, -0.9375, -0.8125, -0.6875, -0.5625, -0.4375, -0.3125, -0.1875, 0, 0.1875,  0.3125,  0.4375,  0.5625,  0.6875,  0.8125,  0.9375,  1.0625 };   
const int MSswitchNow = 20;                                     // threshold for hysteresis 

// generig definitions for debug
//#define DEBUG_MS 1                                            // debuging flag for Multistate state output

void setup_MultiSwitch() {
/* 
  pinMode(msPin_a, OUTPUT);                                     // uncomment because function in other SW module
  pinMode(msPin_b, OUTPUT);                                     // uncomment because function in other SW module
  pinMode(msPin_c, OUTPUT);                                     // uncomment because function in other SW module
  pinMode(msPin_d, OUTPUT);                                     // uncomment because function in other SW module */
}

void close_MultiSwitch() {
/*
 digitalWrite(msPin_a, LOW);                                    // close all digital pins 
 digitalWrite(msPin_b, LOW); 
 digitalWrite(msPin_c, LOW); 
 digitalWrite(msPin_d, LOW);                                    */

  for ( int i = 0; i <4 ; i++) {
    switchOn[i] = false;                                        // and switch of internal sound functions        
  }
}                                                               // function close_MultiSwitch closed

void run_MultiSwitch() {                                        // function to check states of the multiswitch and activate the pinouts if needed

   int S_state = 0;                                             // value for state of switches (0 to 15)

  for (int i = 0; i <= 15; i++) {                               // precalculate switchstate via the thresholds
    if (RC_MultiSwitch > MS_threshold[i] && RC_MultiSwitch <= MS_threshold[i + 1]) {
      S_state = i;                                              // correct state for this value found  
    }
  }

  if ( S_state != last_state) {                                 // check if switches have changed
    #ifdef DEBUG_MS  
      Serial.print("MS_switch ongoing: ");Serial.print(S_state);Serial.print(" changecount: ");Serial.print(hysteresis);         
    #endif
    if (hysteresis >= MSswitchNow) {
      last_state = S_state;                                     // yes we change 
      hysteresis = 0;
    }
    else {
      S_state = last_state;                                     // hysteresis is running, we change later
      hysteresis++; 
    }
  }
  else {                                                        // no change triggered
    hysteresis = 0;                                             
    #ifdef DEBUG_MS                                             // show DEBUG_MS data if needed
      Serial.print("MS_switch no change: ");Serial.print(S_state);
      Serial.print(" MSch");Serial.print(": ");
      Serial.print(RC_MultiSwitch);Serial.print(" ->  ");
      for (int i = 3; i>-1; i--){
        Serial.print(switchOn[i]);Serial.print(",");
      }                   
      Serial.println();
    #endif
    return;                                                     // return - state updated not needed 
  }                                      
   
  if (S_state > 7) {                                            // check state of highest switch
    // digitalWrite(msPin_d, HIGH);                             // and activate pinout
    S_state = S_state - 8;                                      // get ready for next switch 
    switchOn[3] = true;                                         // and activate soft-switch
    #ifdef DEBUG_MS 
      Serial.print(" T,");
    #endif
  }
  else {
    // digitalWrite(msPin_d, LOW);                              // switch is off -> deactivate pinout
    switchOn[3] = false;                                        // and soft-switch
    #ifdef DEBUG_MS 
      Serial.print(" F,");
    #endif    
  }
     
  if (S_state > 3) {                                            // check state of next switch
    // digitalWrite(msPin_c, HIGH);                             // and activate pinout
    S_state = S_state - 4;
    switchOn[2] = true;                                         // and activate soft-switch
    #ifdef DEBUG_MS 
      Serial.print(" T,");
    #endif
  }
  else {
    // digitalWrite(msPin_c, LOW);                              // switch is off -> deactivate pinout
    switchOn[2] = false;                                        // and soft-switch
    #ifdef DEBUG_MS 
      Serial.print(" F,");       
    #endif    
  }

  if (S_state > 1) {                                            // check state of next switch                            
    // digitalWrite(msPin_b, HIGH);                             // and activate pinout
    S_state = S_state - 2;
    switchOn[1] = true;                                         // and activate soft-switch
    #ifdef DEBUG_MS 
      Serial.print(" T,");       
    #endif
  }
  else {
    // digitalWrite(msPin_b, LOW);                              // switch is off -> deactivate pinout
    switchOn[1] = false;                                        // and soft-switch
    #ifdef DEBUG_MS 
      Serial.print(" F,");    
    #endif    
  }

  if (S_state > 0) {                                            // check state of lowest switch 
    // digitalWrite(msPin_a, HIGH);                             // and activate pinout
    switchOn[0] = true;                                         // and activate soft-switch
    #ifdef DEBUG_MS 
      Serial.print(" T,");        
    #endif
  }
  else {
    // digitalWrite(msPin_a, LOW);                              // switch is off -> deactivate pinout
    switchOn[0] = false;                                        // and soft-switch
    #ifdef DEBUG_MS 
      Serial.print(" F:");
    #endif    
  }                                                             // end of switch state check
  #ifdef DEBUG_MS                                               // show DEBUG_MS data if needed
    Serial.print(" MSch");Serial.print(": ");
    Serial.print(RC_MultiSwitch);Serial.print(" ->  ");
    for (int i = 3; i>-1; i--){
      Serial.print(switchOn[i]);Serial.print(",");
    }                   
    Serial.println();
  #endif
}                                                              // function run_MultiSwitch closed
